/*
 * This source code is made available under the terms of the
 * Common Public License Version 1.0 distibuted along with the
 * source code package.
 */
package de.uka.cmtm.serviceregistry.publish.uddi;

import java.net.URL;

import org.uddi4j.client.UDDIProxy;
import org.uddi4j.response.AuthToken;

import de.uka.cmtm.serviceregistry.publish.uddi.wsdl.WsdlPublisherV01_08;
import de.uka.cmtm.serviceregistry.publish.uddi.wsdls.SaWsdlPublisherV01_08;

/**
 * An Enum containing mapping versions between WSDL and UDDI
 * 
 * @author tilmann
 */
enum MappingVersion {

	/**
	 * UDDI.org best practices mapping version 1.08
	 */
	v01_08;
}

/**
 * This class can be used to create instances of WSDLPublisher.
 * 
 * @author tilmann
 */
public class WsdlPublisherFactory {

	private MappingVersion mappingVersion = MappingVersion.v01_08;

	private boolean saWsdlPublishingCapable = true;

	private String uddiInquiryUrl;

	private String uddiPublishUrl;

	private String uddiUser;

	private String uddiPassword;

	private String upperServiceOntologyUrl;

	/**
	 * @return the upperServiceOntologyUrl
	 */
	public String getUpperServiceOntologyUrl() {
		return upperServiceOntologyUrl;
	}

	/**
	 * @param upperServiceOntologyUrl
	 *            the upperServiceOntologyUrl to set
	 */
	public void setUpperServiceOntologyUrl(String upperServiceOntologyUrl) {
		this.upperServiceOntologyUrl = upperServiceOntologyUrl;
	}

	/**
	 * @return the mappingVersion
	 */
	public MappingVersion getMappingVersion() {
		return mappingVersion;
	}

	/**
	 * @param mappingVersion
	 *            the mappingVersion to set
	 */
	public void setMappingVersion(MappingVersion mappingVersion) {
		this.mappingVersion = mappingVersion;
	}

	/**
	 * @return the uddiInquiryUrl
	 */
	public String getUddiInquiryUrl() {
		return uddiInquiryUrl;
	}

	/**
	 * @param uddiInquiryUrl
	 *            the uddiInquiryUrl to set
	 */
	public void setUddiInquiryUrl(String uddiInquiryUrl) {
		this.uddiInquiryUrl = uddiInquiryUrl;
	}

	/**
	 * @return the uddiPassword
	 */
	public String getUddiPassword() {
		return uddiPassword;
	}

	/**
	 * @param uddiPassword
	 *            the uddiPassword to set
	 */
	public void setUddiPassword(String uddiPassword) {
		this.uddiPassword = uddiPassword;
	}

	/**
	 * @return the uddiPublishUrl
	 */
	public String getUddiPublishUrl() {
		return uddiPublishUrl;
	}

	/**
	 * @param uddiPublishUrl
	 *            the uddiPublishUrl to set
	 */
	public void setUddiPublishUrl(String uddiPublishUrl) {
		this.uddiPublishUrl = uddiPublishUrl;
	}

	/**
	 * @return the uddiUser
	 */
	public String getUddiUser() {
		return uddiUser;
	}

	/**
	 * @param uddiUser
	 *            the uddiUser to set
	 */
	public void setUddiUser(String uddiUser) {
		this.uddiUser = uddiUser;
	}

	/**
	 * @return true if instances created by the facory are capable of publishing
	 *         SA-WSDL files
	 */
	public boolean isSaWsdlPublishingCapable() {
		return saWsdlPublishingCapable;
	}

	/**
	 * @param saWsdlPublishingCapable
	 *            set if instances created by the factory are to be capable of
	 *            publishing SA-WSDL files.
	 */
	public void setSaWsdlPublishingCapable(boolean saWsdlPublishingCapable) {
		this.saWsdlPublishingCapable = saWsdlPublishingCapable;
	}

	public WsdlPublisher createWsdlPublisher() throws InstantiationException {

		// Check state of attributes
		if (uddiPublishUrl == null)
			throw new IllegalStateException(
					"UddiPublishUrl must be set prior to a call to createWsdlPublisher!");
		if (uddiInquiryUrl == null)
			throw new IllegalStateException(
					"UddiInquiryUrl must be set prior to a call to createWsdlPublisher!");
		if (uddiPassword == null)
			throw new IllegalStateException(
					"UddiPassword must be set prior to a call to createWsdlPublisher!");
		if (uddiUser == null)
			throw new IllegalStateException(
					"UddiUser must be set prior to a call to createWsdlPublisher!");

		UDDIProxy uddiProxy;
		try {
			uddiProxy = new UDDIProxy(new URL(uddiInquiryUrl), new URL(
					uddiPublishUrl));
			AuthToken authToken = uddiProxy.get_authToken(uddiUser,
					uddiPassword);

			switch (mappingVersion) {
			case v01_08: {
				if (isSaWsdlPublishingCapable()) {
					if (upperServiceOntologyUrl == null)
						throw new IllegalStateException(
								"UpperServiceOntologyUrl must be set before a wsdls-bupblishing capable WsdlPublisher can be created.");
					return new SaWsdlPublisherV01_08(uddiProxy, authToken,
							upperServiceOntologyUrl);
				} else
					return new WsdlPublisherV01_08(uddiProxy, authToken);
			}
			}
		} catch (Exception e) {
			InstantiationException ie = new InstantiationException(
					"UDDI connection problem.");
			ie.initCause(e);
			throw ie;
		}

		return null;
	}

}
