package edu.stanford.smi.protegex.owl.ui.components.triples;

import com.hp.hpl.jena.datatypes.xsd.impl.XMLLiteralType;
import edu.stanford.smi.protege.ui.FrameComparator;
import edu.stanford.smi.protegex.owl.model.*;
import edu.stanford.smi.protegex.owl.model.event.PropertyValueAdapter;
import edu.stanford.smi.protegex.owl.model.event.PropertyValueListener;
import edu.stanford.smi.protegex.owl.model.impl.DefaultRDFSLiteral;
import edu.stanford.smi.protegex.owl.model.impl.XMLSchemaDatatypes;
import edu.stanford.smi.protegex.owl.ui.ProtegeUI;
import edu.stanford.smi.protegex.owl.ui.metadata.AnnotationsWidgetPlugin;

import javax.swing.*;
import javax.swing.table.AbstractTableModel;
import java.net.URI;
import java.util.*;

/**
 * A TableModel to display property values for a given subject.
 * Each row in this table represents a single triple.
 *
 * @author Holger Knublauch  <holger@knublauch.com>
 */
public class TriplesTableModel extends AbstractTableModel {

    /**
     * The list of Properties currently displayed
     */
    private ArrayList properties = new ArrayList();

    /**
     * The resource being annotated
     */
    private RDFResource subject;

    private JTable table;

    /**
     * A FrameListener that detects any changes in own slot values of annotation
     * properties and then updates the table model accordingly.
     */
    private PropertyValueListener valueListener = new PropertyValueAdapter() {
        public void propertyValueChanged(RDFResource resource, RDFProperty property, Collection oldValues) {
            if (isRelevantProperty(property)) {
                updateValues();
            }
        }
    };


    /**
     * The values currently displayed in each row (either plain values or RDFObjects)
     */
    private ArrayList values = new ArrayList();

    public final static int COL_PROPERTY = 0;

    public final static int COL_VALUE = 1;


    public TriplesTableModel() {
    }


    public TriplesTableModel(RDFResource subject) {
        this.subject = subject;
        if (subject != null) {
            subject.addPropertyValueListener(valueListener);
        }
        refill();
    }


    public int addRow(RDFProperty property) {
        Object value = createDefaultValue(property);
        if (value != null) {
            return addRow(property, value);
        }
        else {
            return -1;
        }
    }


    public int addRow(RDFProperty property, Object value) {
        if (!subject.hasPropertyValue(property, value)) {
            subject.addPropertyValue(property, value);
        }
        return getPropertyValueRow(property, value);
    }


    private Object createDefaultValue(RDFProperty property) {
        for (Iterator it = TriplesComponent.plugins(); it.hasNext();) {
            AnnotationsWidgetPlugin plugin = (AnnotationsWidgetPlugin) it.next();
            if (plugin.canEdit(subject, property, null)) {
                Object defaultValue = plugin.createDefaultValue((RDFResource) subject, property);
                if (defaultValue != null) {
                    return defaultValue;
                }
            }
        }
        RDFResource range = property.getRange();
        if (range instanceof RDFSDatatype) {
            return ((RDFSDatatype) range).getDefaultValue();
        }
        else if (range instanceof OWLDataRange) {
            OWLDataRange dataRange = (OWLDataRange) range;
            RDFList oneOf = dataRange.getOneOf();
            Object firstValue = null;
            if (oneOf != null) {
                Collection values = oneOf.getValues();
                if (!values.isEmpty()) {
                    firstValue = values.iterator().next();
                }
            }
            return firstValue;
        }
        else if (property instanceof OWLObjectProperty) {
            return null;
        }
        else {
            return "";
        }
    }


    private Object createNewValue(RDFProperty property, String text, String language) {
        Object newValue = null;
        if (language == null || language.trim().length() == 0) {
            newValue = text;
        }
        else {
            newValue = property.getOWLModel().createRDFSLiteralOrString(text, language);
        }
        return newValue;
    }


    public void deleteRow(int rowIndex) {
        RDFProperty property = getPredicate(rowIndex);
        Object value = getValue(rowIndex);
        subject.removePropertyValue(property, value);
    }


    public void dispose() {
        if (subject != null) {
            subject.removePropertyValueListener(valueListener);
        }
    }


    public Class getColumnClass(int column) {
        if (column == COL_PROPERTY) {
            return RDFProperty.class;
        }
        else if (column == COL_VALUE) {
            return Object.class;
        }
        else if (hasTypeColumn() && column == COL_VALUE + 1) {
            return RDFResource.class;
        }
        else {
            return String.class;
        }
    }


    public int getColumnCount() {
        return hasTypeColumn() ? 4 : 3;
    }


    public String getColumnName(int column) {
        if (column == COL_PROPERTY) {
            return "Property";
        }
        else if (column == COL_VALUE) {
            return "Value";
        }
        else if (isTypeColumn(column)) {
            return "Type";
        }
        else {
            return "Lang";
        }
    }


    public Object getDisplayValue(int rowIndex) {
        Object value = values.get(rowIndex);
        if (value instanceof RDFSLiteral) {
            return ((RDFSLiteral) value).getString();
        }
        return value;
    }


    public String getLanguage(int row) {
        Object value = getValue(row);
        if (value instanceof RDFSLiteral) {
            return ((RDFSLiteral) value).getLanguage();
        }
        return null;
    }


    public OWLModel getOWLModel() {
        return subject.getOWLModel();
    }


    public RDFProperty getPredicate(int rowIndex) {
        return (RDFProperty) properties.get(rowIndex);
    }


    public int getPropertyValueRow(RDFProperty property, Object value) {
        for (int i = 0; i < properties.size(); i++) {
            RDFProperty s = getPredicate(i);
            if (s.equals(property)) {
                if (value == null && getValue(i) == null) {
                    return i;
                }
                else if (value != null && value.equals(getValue(i))) {
                    return i;
                }
            }
        }
        return -1;
    }


    protected Collection getRelevantProperties() {
        OWLModel owlModel = subject.getOWLModel();
	    Collection props = new HashSet();
	    for(Iterator it = owlModel.getRDFProperties().iterator(); it.hasNext();) {
		    RDFProperty curProp = (RDFProperty) it.next();
		    if(curProp.isVisible()) {
			    props.add(curProp);
		    }
	    }
	    props.add(owlModel.getRDFSIsDefinedByProperty());
	    props.add(owlModel.getRDFSLabelProperty());
	    props.add(owlModel.getRDFProperty(RDFSNames.Slot.SEE_ALSO));
	    props.add(owlModel.getOWLSameAsProperty());
	    props.add(owlModel.getOWLEquivalentPropertyProperty());
	    props.add(owlModel.getOWLDifferentFromProperty());
	    props.add(owlModel.getOWLVersionInfoProperty());
        //return subject.getRDFProperties();
        return props;
    }


    public int getRowCount() {
        return properties.size();
    }


    public RDFResource getSubject() {
        return subject;
    }


    public Object getValueAt(int rowIndex, int columnIndex) {
        if (columnIndex == COL_PROPERTY) {
            return getPredicate(rowIndex);
        }
        else if (columnIndex == COL_VALUE) {
            return getDisplayValue(rowIndex);
        }
        else if (isTypeColumn(columnIndex)) {
            Object value = getValue(rowIndex);
            if (value instanceof RDFResource) {
                return ((RDFResource) value).getRDFType();
            }
            else if (value instanceof RDFSLiteral) {
                return ((RDFSLiteral) value).getDatatype();
            }
            else {
                RDFSLiteral literal = DefaultRDFSLiteral.create(getOWLModel(), value);
                return literal.getDatatype();
            }
        }
        else {
            return getLanguage(rowIndex);
        }
    }


    public Object getValue(int rowIndex) {
        return values.get(rowIndex);
    }


    protected boolean hasTypeColumn() {
        return true;
    }


    public boolean isCellEditable(int rowIndex, int columnIndex) {
        if (columnIndex != COL_PROPERTY) {
            RDFProperty property = getPredicate(rowIndex);
            if (columnIndex == COL_VALUE) {
                Object value = getValue(rowIndex);
                for (Iterator it = TriplesComponent.plugins(); it.hasNext();) {
                    AnnotationsWidgetPlugin plugin = (AnnotationsWidgetPlugin) it.next();
                    if (plugin.canEdit(subject, property, value)) {
                        return false;
                    }
                }
                OWLModel owlModel = property.getOWLModel();
                return (property instanceof OWLDatatypeProperty &&
                        !property.isReadOnly() &&
                        owlModel.getTripleStoreModel().isActiveTriple(subject, property, value)) ||
                       getDefaultProperties().contains(property) && value == null;
            }
            else if (isTypeColumn(columnIndex)) {
                return getValueAt(rowIndex, columnIndex) instanceof RDFSDatatype;
            }
            else {
                return property instanceof OWLDatatypeProperty &&
                        property.getOWLModel().getXSDstring().equals(property.getRange()) &&
                        !property.isReadOnly();
            }
        }
        else {
            return false;
        }
    }


    public boolean isDeleteEnabled(int row) {
        RDFProperty predicate = getPredicate(row);
        Object object = getValue(row);
        return edu.stanford.smi.protegex.owl.ui.actions.triple.DeleteTripleAction.isSuitable(subject, predicate, object);
    }


    public static boolean isInvalidXMLLiteral(RDFProperty property, Object value) {
        if (XMLSchemaDatatypes.isXMLLiteralSlot(property) && value instanceof String) {
            if (!XMLLiteralType.theXMLLiteralType.isValid((String) value)) {
                ProtegeUI.getModalDialogFactory().showErrorMessageDialog(property.getOWLModel(),
                        "This value is not a valid XML literal:\n" + value);
                return true;
            }
        }
        return false;
    }


    protected boolean isRelevantProperty(RDFProperty property) {
        return true;
    }


    private boolean isTypeColumn(int column) {
        return hasTypeColumn() && column == COL_VALUE + 1;
    }


    private void refill() {
        if (subject != null) {
            Collection properties = getRelevantProperties();
            RDFProperty[] ss = (RDFProperty[]) properties.toArray(new RDFProperty[0]);
            Arrays.sort(ss, new FrameComparator());
            for (int i = 0; i < ss.length; i++) {
                RDFProperty property = ss[i];
		        Collection values = subject.getPropertyValues(property);
		        for (Iterator it = values.iterator(); it.hasNext();) {
		            Object value = it.next();
		            this.properties.add(property);
		            this.values.add(value);
		        }
            }
	        for(Iterator it = getDefaultProperties().iterator(); it.hasNext(); ) {
		        RDFProperty curProp = (RDFProperty) it.next();
		        if(this.properties.contains(curProp) == false) {
			        this.properties.add(0, curProp);
			        this.values.add(0, null);
		        }
	        }
        }
    }

    private void setDatatype(int row, RDFSDatatype datatype) {
        Object oldValue = getValue(row);
        final String lexicalValue = oldValue.toString();
        RDFSLiteral newValue = datatype.getOWLModel().createRDFSLiteral(lexicalValue, datatype);
        RDFProperty predicate = getPredicate(row);
        subject.removePropertyValue(predicate, oldValue);
        subject.addPropertyValue(predicate, newValue);
    }


    private Object setLanguage(int row, String newLanguage) {
        RDFProperty property = getPredicate(row);
        String text = (String) getDisplayValue(row);
        Object newValue = createNewValue(property, text, newLanguage);
        if (!subject.getPropertyValues(property).contains(newValue)) {
            Object oldValue = getValue(row);
            subject.removePropertyValue(property, oldValue);
            subject.addPropertyValue(property, newValue);
        }
        return newValue;
    }


    public void setSubject(RDFResource instance) {
        RDFResource oldSubject = this.subject;
        if (oldSubject != null) {
            oldSubject.removePropertyValueListener(valueListener);
        }
        this.subject = instance;
        properties.clear();
        values.clear();
        if (instance != null) {
            instance.addPropertyValueListener(valueListener);
            refill();
        }
        fireTableDataChanged();
    }


    public Object setValue(Object aValue, int row) {
        RDFProperty property = getPredicate(row);
        Object oldValue = getValue(row);
        if (oldValue == null || !oldValue.equals(aValue)) {
            RDFResource range = property.getRange();
            try {
                Object newValue;
                String str = aValue.toString();
                OWLModel owlModel = property.getOWLModel();
                if (owlModel.getOWLOntologyProperties().contains(property)) {
                    String message = str + " is not a valid URI.";
                    try {
                        new URI(str);
                    }
                    catch (Exception ex) {
                        ProtegeUI.getModalDialogFactory().showErrorMessageDialog(owlModel, message);
                        return oldValue;
                    }
                    if (!str.startsWith("http://") && !str.startsWith("file:")) {
                        ProtegeUI.getModalDialogFactory().showErrorMessageDialog(owlModel, message);
                        return oldValue;
                    }
                }
                String lang = getLanguage(row);
                if(aValue instanceof RDFSLiteral && ((RDFSLiteral)aValue).getLanguage() != null) {
                    newValue = aValue;
                }
                else if (lang != null) {
                    newValue = createNewValue(property, str, lang);
                }
                else if (oldValue instanceof RDFSLiteral) {
                    RDFSLiteral oldLiteral = (RDFSLiteral) oldValue;
                    newValue = getOWLModel().createRDFSLiteral(str, oldLiteral.getDatatype());
                }
                else if (oldValue instanceof Boolean) {
                    newValue = Boolean.valueOf(str.equals("true"));
                }
                else if (oldValue instanceof Float) {
                    newValue = Float.valueOf(str);
                }
                else if (oldValue instanceof Integer) {
                    newValue = Integer.valueOf(str);
                }
                else {
                    newValue = str;
                }
                if (!subject.getPropertyValues(property).contains(newValue)) {
                    subject.removePropertyValue(property, oldValue);
                    subject.addPropertyValue(property, newValue);
                }
                return newValue;
            }
            catch (NumberFormatException ex) {
                // Ignore illegal number format
            }
        }
        return oldValue;
    }


    public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
        setValueAndGetIt(aValue, rowIndex, columnIndex);
    }


    public Object setValueAndGetIt(Object value, int row, int col) {
        if (col == COL_PROPERTY) {
            return null;
        }
        if (col == COL_VALUE) {
            return setValue(value, row);
        }
        else if (isTypeColumn(col)) {
            if (value instanceof RDFSDatatype) {
                RDFSDatatype datatype = (RDFSDatatype) value;
                setDatatype(row, datatype);
                return datatype;
            }
            else {
                return null;
            }
        }
        else {
            return setLanguage(row, (String) value);
        }
    }


    /**
     * Removes all rows containing a given property and then re-adds them with the
     * most recent values.
     */
    void updateValues() {
        int index = -1;
        if (table != null) {
            index = table.getSelectedRow();
        }
        properties.clear();
        values.clear();
        refill();
        fireTableDataChanged();
        if (table != null && index >= 0 && index < getRowCount()) {
            table.getSelectionModel().setSelectionInterval(index, index);
        }
    }


    public void setTable(TriplesTable table) {
        this.table = table;
    }

	public Collection getDefaultProperties() {
		return Collections.EMPTY_LIST;
	}
}
